"""Service for managing task assignments to employees."""

from __future__ import annotations

from datetime import date
from typing import List, Optional

from ca_task_manager.repositories.assignment_repo import AssignmentRepository
from ca_task_manager.repositories.task_repo import TaskRepository
from ca_task_manager.services.task_service import TaskService
from ca_task_manager.models.assignment import Assignment, AssignmentStatus
from ca_task_manager.models.task import Recurrence


class AssignmentService:
    def __init__(
        self,
        assignment_repo: Optional[AssignmentRepository] = None,
        task_repo: Optional[TaskRepository] = None,
        task_service: Optional[TaskService] = None,
    ):
        self.repo = assignment_repo or AssignmentRepository()
        self.task_repo = task_repo or TaskRepository()
        self.task_service = task_service or TaskService(self.task_repo)

    def assign_task(self, task_id: int, employee_ids: List[int], assigned_by: int, assigned_on: Optional[date] = None) -> List[int]:
        """Assign a task to multiple employees.  Returns list of assignment IDs."""
        assigned_on = assigned_on or date.today()
        ids = []
        for emp_id in employee_ids:
            assignment = Assignment(
                task_id=task_id,
                employee_id=emp_id,
                assigned_on=assigned_on,
                assigned_by=assigned_by,
            )
            data = assignment.dict()
            # Convert assigned_on to ISO string for JSON
            data["assigned_on"] = assigned_on.isoformat()
            assignment_id = self.repo.create_assignment(Assignment(**data))
            ids.append(assignment_id)
        return ids

    def update_assignment(self, assignment_id: int, updates: dict) -> bool:
        """Update an assignment.  Handles completion events for recurring tasks."""
        # If percent_complete set to 100 or status Completed, set status accordingly
        status = updates.get("status")
        percent = updates.get("percent_complete")
        if percent is not None and percent >= 100:
            updates["status"] = AssignmentStatus.COMPLETED.value
        if status == AssignmentStatus.COMPLETED.value:
            # Set actual_completion_date if not set
            from datetime import date as d

            updates.setdefault("actual_completion_date", d.today().isoformat())
        # Apply update
        result = self.repo.update_assignment(assignment_id, updates)
        if not result:
            return False
        # If assignment is completed, and the underlying task is recurring, schedule next instance
        updated_assignment = self.repo.get(assignment_id)
        if updated_assignment and updated_assignment.get("status") == AssignmentStatus.COMPLETED.value:
            task = self.task_repo.get(updated_assignment.get("task_id"))
            if task:
                recurrence = task.get("recurrence", Recurrence.NONE)
                try:
                    rec_enum = Recurrence(recurrence) if isinstance(recurrence, str) else recurrence
                except ValueError:
                    rec_enum = Recurrence.NONE
                if rec_enum != Recurrence.NONE:
                    # create next recurring task
                    new_task_id = self.task_service.handle_recurrence_on_completion(task)
                    # Optionally auto-assign the new task to same employees
                    if new_task_id:
                        emp_id = updated_assignment.get("employee_id")
                        assigned_by = updated_assignment.get("assigned_by")
                        self.assign_task(new_task_id, [emp_id], assigned_by)
        return True

    def get_assignments_for_employee(self, employee_id: int) -> List[dict]:
        return self.repo.assignments_for_employee(employee_id)

    def get_assignments_for_task(self, task_id: int) -> List[dict]:
        return self.repo.assignments_for_task(task_id)

    def get_due_assignments(self, start_date: date, end_date: date) -> List[dict]:
        """Return assignments whose tasks' due_date falls within given window."""
        due_assignments: List[dict] = []
        tasks_within = self.task_repo.tasks_due_within(start_date, end_date)
        task_ids = {t["id"] for t in tasks_within}
        for assignment in self.repo.pending_assignments():
            if assignment.get("task_id") in task_ids:
                due_assignments.append(assignment)
        return due_assignments